<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Mail\AdminMail;
use App\Mail\OrderMail;
use App\Models\BlockedIp;
use App\Models\GmailSmtp;
use App\Models\Order;
use App\Models\Payment;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class PaymentController extends Controller
{
    public function payment(Request $request)
    {
        session([
            'checkoutData' => $request->all(),
        ]);
        
        $data = $request->all();

        $ip = $request->ip();
        $userAgent = $request->userAgent();

        $blocked = BlockedIp::where('ip_address', $ip)
            ->where('user_agent', $userAgent)
            ->first();

        if ($blocked) {
            return abort(403, 'Your IP is blocked.');
        }

        $twoMinAgo   = now()->subMinutes(2);
        $tenMinAgo   = now()->subMinutes(10);
        $todayStart  = now()->startOfDay();

        $orders2Min = Order::where('ip_address', $ip)
            ->where('user_agent', $userAgent)
            ->where('created_at', '>=', $twoMinAgo)
            ->count();

        $orders10Min = Order::where('ip_address', $ip)
            ->where('user_agent', $userAgent)
            ->where('created_at', '>=', $tenMinAgo)
            ->count();

        $ordersToday = Order::where('ip_address', $ip)
            ->where('user_agent', $userAgent)
            ->where('created_at', '>=', $todayStart)
            ->count();

        $block = false;
        $reason = '';

        if ($orders2Min >= 5) {
            $block = true;
            $reason .= "More than 5 orders in 2 minutes. ";
        }

        if ($orders10Min >= 20) {
            $block = true;
            $reason .= "More than 20 orders in 10 minutes. ";
        }

        if ($ordersToday >= 20) {
            $block = true;
            $reason .= "More than 20 orders today. ";
        }

        if ($block) {
            BlockedIp::updateOrCreate(
                ['ip_address' => $ip, 'user_agent' => $userAgent],
                ['reason' => $reason]
            );

            return abort(403, 'Your IP is temporarily blocked due to suspicious activity.');
        }

        $paymentMethod = $request->payment_method;

        $orderController = new OrderController;

        return $orderController->createOrder($request, $paymentMethod);
    }

    public function paymentSuccess(Request $request)
    {
        $request->validate([
            'order_id' => 'required|integer|exists:orders,id',
            'payment_method' => 'required|string',
            'sender_number' => 'required|string',
            'transaction_id' => 'required|string',
        ]);

        $orderId = $request->order_id;

        $order = Order::findOrFail($orderId);

        $payment = Payment::create([
            'order_id' => $order->id,
            'payment_method' => $request->payment_method,
            'amount' => $order->total,
            'sender_number' => $request->sender_number,
            'transaction_id' => $request->transaction_id,
            'status' => 'success',
        ]);

        return redirect()->route('order.success', [
            'invoice_id' => $order->invoice_id,
            'amount' => $order->total,
            'order' => $order,
            'payment' => $payment,
            'method' => $request->payment_method == 'cod' ? 'Cash on Delivery' : $request->payment_method,
        ])->with('success', 'Your order has been placed successfully!');
    }

    public function paymentFail()
    {
        return redirect()->route('checkout.index')->with('error', 'Payment failed! Please try again.');
    }
}
